<?php
/**
 * Plugin Name: DigiKash WooCommerce Gateway
 * Description: DigiKash payment gateway for WooCommerce
 * Version: 2.7.0
 * Author: DigiKash Team
 */

if (!defined('ABSPATH')) exit;

// Debug: Log when plugin loads
error_log('DigiKash Plugin: Loading...');

add_action('plugins_loaded', 'digikash_gateway_init', 11);
function digikash_gateway_init() {
    error_log('DigiKash Plugin: plugins_loaded hook fired');
    
    if (!class_exists('WC_Payment_Gateway')) {
        error_log('DigiKash Plugin: WC_Payment_Gateway class not found');
        return;
    }
    
    error_log('DigiKash Plugin: WC_Payment_Gateway class found');
    
    if (!class_exists('WC_Gateway_DigiKash')) {
        error_log('DigiKash Plugin: Defining gateway class');
        
        class WC_Gateway_DigiKash extends WC_Payment_Gateway {
            public $api_base_url;
            public $merchant_id;
            public $api_key;
            public $webhook_secret;
            public $testmode;

            public function __construct() {
                error_log('DigiKash Gateway: Constructor called');
                
                $this->id = 'digikash';
                $this->has_fields = true;
                $this->method_title = 'DigiKash Payment Gateway';
                $this->method_description = 'Accept payments through DigiKash secure payment gateway.';
                $this->supports = array('products');
                
                $this->init_form_fields();
                $this->init_settings();
                
                $this->title = $this->get_option('title', 'DigiKash');
                $this->description = $this->get_option('description', 'Pay securely using DigiKash.');
                $this->api_base_url = $this->get_option('api_base_url', 'https://digikash.coevs.com');
                $this->merchant_id = $this->get_option('merchant_id');
                $this->api_key = $this->get_option('api_key');
                $this->webhook_secret = $this->get_option('webhook_secret');
                $this->testmode = 'yes' === $this->get_option('testmode', 'yes');
                
                error_log('DigiKash Gateway: Constructor completed - ID: ' . $this->id);
                error_log('DigiKash Gateway: Settings - enabled=' . $this->get_option('enabled') . ', merchant_id=' . $this->merchant_id . ', api_key=' . (empty($this->api_key) ? 'EMPTY' : 'SET'));
                
                add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
                
                // Register webhook handler
                add_action('woocommerce_api_digikash_webhook', array($this, 'webhook_handler'));
            }

            public function init_form_fields() {
                $this->form_fields = array(
                    'enabled' => array(
                        'title' => 'Enable/Disable',
                        'type' => 'checkbox',
                        'label' => 'Enable DigiKash Payment Gateway',
                        'default' => 'yes'
                    ),
                    'title' => array(
                        'title' => 'Title',
                        'type' => 'text',
                        'description' => 'Payment method title that customers see during checkout.',
                        'default' => 'DigiKash',
                        'desc_tip' => true
                    ),
                    'description' => array(
                        'title' => 'Description',
                        'type' => 'textarea',
                        'description' => 'Payment method description that customers see during checkout.',
                        'default' => 'Pay securely using DigiKash.',
                        'desc_tip' => true
                    ),
                    'api_base_url' => array(
                        'title' => 'API Base URL',
                        'type' => 'url',
                        'description' => 'Enter your DigiKash API base URL.',
                        'default' => 'https://digikash.coevs.com',
                        'desc_tip' => true
                    ),
                    'merchant_id' => array(
                        'title' => 'Merchant ID',
                        'type' => 'text',
                        'description' => 'Get your Merchant ID from DigiKash dashboard.',
                        'desc_tip' => true
                    ),
                    'api_key' => array(
                        'title' => 'API Key',
                        'type' => 'password',
                        'description' => 'Get your API Key from DigiKash dashboard.',
                        'desc_tip' => true
                    ),
                    'webhook_secret' => array(
                        'title' => 'Webhook Secret Key',
                        'type' => 'password',
                        'description' => 'Enter your Webhook Secret Key for secure webhook verification. Get this from DigiKash dashboard.',
                        'desc_tip' => true
                    ),
                    'testmode' => array(
                        'title' => 'Test Mode',
                        'label' => 'Enable Test Mode',
                        'type' => 'checkbox',
                        'description' => 'Place the payment gateway in test mode using sandbox environment.',
                        'default' => 'yes',
                        'desc_tip' => true
                    )
                );
            }

            public function is_available() {
                error_log('DigiKash Gateway: is_available() called');
                error_log('DigiKash Gateway: enabled setting = ' . $this->get_option('enabled'));
                error_log('DigiKash Gateway: merchant_id = ' . $this->merchant_id);
                error_log('DigiKash Gateway: api_key = ' . (empty($this->api_key) ? 'EMPTY' : 'SET'));
                
                if ('yes' !== $this->get_option('enabled')) {
                    error_log('DigiKash Gateway: Gateway disabled in settings');
                    return false;
                }
                
                if (empty($this->merchant_id) || empty($this->api_key) || empty($this->api_base_url)) {
                    error_log('DigiKash Gateway: Required fields missing');
                    return false;
                }
                
                $parent_available = parent::is_available();
                error_log('DigiKash Gateway: parent::is_available() = ' . ($parent_available ? 'true' : 'false'));
                
                // FORCE TRUE for debugging
                error_log('DigiKash Gateway: FORCING TRUE for debugging');
                return true;
            }

            public function payment_fields() {
                $site_info = $this->get_site_info();
                
                echo '<div class="digikash-payment-container" style="
                    background: linear-gradient(135deg, #f8f9fa 0%, #ffffff 100%);
                    border: 2px solid #e9ecef;
                    border-radius: 12px;
                    padding: 24px;
                    margin: 16px 0;
                    position: relative;
                    overflow: hidden;
                    transition: all 0.3s ease;
                    box-shadow: 0 4px 12px rgba(0,0,0,0.05);
                " onmouseover="this.style.boxShadow=\'0 8px 25px rgba(0,0,0,0.1)\'" onmouseout="this.style.boxShadow=\'0 4px 12px rgba(0,0,0,0.05)\'">';
                
                // Modern header with logo and branding
                echo '<div class="digikash-header" style="
                    display: flex;
                    align-items: center;
                    gap: 16px;
                    margin-bottom: 20px;
                    padding-bottom: 16px;
                    border-bottom: 1px solid #e9ecef;
                ">';
                
                if (!empty($site_info['logo'])) {
                    echo '<div class="digikash-logo" style="
                        width: 48px;
                        height: 48px;
                        border-radius: 8px;
                        overflow: hidden;
                        background: #ffffff;
                        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
                        display: flex;
                        align-items: center;
                        justify-content: center;
                    ">
                        <img src="' . esc_url($site_info['logo']) . '" alt="DigiKash" style="
                            max-width: 100%;
                            max-height: 100%;
                            object-fit: contain;
                        " />
                    </div>';
                }
                
                echo '<div class="digikash-title-section" style="flex: 1;">';
                echo '<h3 style="
                    margin: 0;
                    font-size: 18px;
                    font-weight: 600;
                    color: #212529;
                    line-height: 1.2;
                ">' . esc_html($site_info['gateway_name']) . '</h3>';
                
                if (!empty($site_info['site_name'])) {
                    echo '<p style="
                        margin: 4px 0 0 0;
                        font-size: 14px;
                        color: #6c757d;
                        font-weight: 500;
                    ">Powered by ' . esc_html($site_info['site_name']) . '</p>';
                }
                echo '</div>';
                
                // Security badge
                echo '<div class="security-badge" style="
                    background: linear-gradient(135deg, #28a745, #20c997);
                    color: white;
                    padding: 8px 12px;
                    border-radius: 20px;
                    font-size: 11px;
                    font-weight: 600;
                    text-transform: uppercase;
                    letter-spacing: 0.5px;
                    box-shadow: 0 2px 6px rgba(40, 167, 69, 0.3);
                ">
                    <i style="margin-right: 4px;">🔒</i> Secure
                </div>';
                
                echo '</div>'; // End header
                
                // Payment description with features
                echo '<div class="digikash-description" style="margin-bottom: 20px;">';
                if (!empty($site_info['description'])) {
                    echo '<p style="
                        margin: 0 0 16px 0;
                        color: #495057;
                        font-size: 15px;
                        line-height: 1.6;
                    ">' . esc_html($site_info['description']) . '</p>';
                }
                
                // Feature badges
                echo '<div class="feature-badges" style="
                    display: flex;
                    flex-wrap: wrap;
                    gap: 8px;
                    margin-bottom: 16px;
                ">';
                
                $features = [
                    ['icon' => '⚡', 'text' => 'Instant Processing'],
                    ['icon' => '🌍', 'text' => 'Global Support'],
                    ['icon' => '📱', 'text' => 'Mobile Ready'],
                    ['icon' => '🛡️', 'text' => 'Bank Level Security']
                ];
                
                foreach ($features as $feature) {
                    echo '<div style="
                        background: #f8f9fa;
                        border: 1px solid #e9ecef;
                        border-radius: 16px;
                        padding: 6px 12px;
                        font-size: 12px;
                        color: #495057;
                        display: flex;
                        align-items: center;
                        gap: 4px;
                        transition: all 0.2s ease;
                    " onmouseover="this.style.background=\'#e9ecef\'" onmouseout="this.style.background=\'#f8f9fa\'">
                        <span>' . $feature['icon'] . '</span>
                        <span>' . $feature['text'] . '</span>
                    </div>';
                }
                echo '</div>';
                echo '</div>';
                
                // Security and SSL info
                if (!empty($site_info['ssl_enabled']) && $site_info['ssl_enabled']) {
                    echo '<div class="ssl-info" style="
                        background: linear-gradient(135deg, #e8f5e8, #f1f9f1);
                        border: 1px solid #d4edda;
                        border-radius: 8px;
                        padding: 12px 16px;
                        margin-bottom: 16px;
                        display: flex;
                        align-items: center;
                        gap: 12px;
                    ">';
                    echo '<div style="
                        width: 32px;
                        height: 32px;
                        background: #28a745;
                        border-radius: 50%;
                        display: flex;
                        align-items: center;
                        justify-content: center;
                        color: white;
                        font-size: 14px;
                        font-weight: bold;
                    ">✓</div>';
                    echo '<div>';
                    echo '<p style="
                        margin: 0;
                        color: #155724;
                        font-size: 14px;
                        font-weight: 600;
                    ">SSL Protected Transaction</p>';
                    echo '<p style="
                        margin: 2px 0 0 0;
                        color: #6c757d;
                        font-size: 12px;
                    ">Your payment information is encrypted and secure</p>';
                    echo '</div>';
                    echo '</div>';
                }
                
                // Environment indicator (only show for test mode)
                if ($this->testmode === 'yes') {
                    echo '<div class="test-mode-alert" style="
                        background: linear-gradient(135deg, #fff3cd, #fef6d3);
                        border: 1px solid #ffeaa7;
                        border-radius: 8px;
                        padding: 12px 16px;
                        margin-bottom: 16px;
                        display: flex;
                        align-items: center;
                        gap: 12px;
                        animation: pulse 2s infinite;
                    ">';
                    echo '<div style="
                        width: 32px;
                        height: 32px;
                        background: #ffc107;
                        border-radius: 50%;
                        display: flex;
                        align-items: center;
                        justify-content: center;
                        color: #212529;
                        font-size: 14px;
                        font-weight: bold;
                    ">⚠</div>';
                    echo '<div>';
                    echo '<p style="
                        margin: 0;
                        color: #856404;
                        font-size: 14px;
                        font-weight: 600;
                    ">Sandbox Environment</p>';
                    echo '<p style="
                        margin: 2px 0 0 0;
                        color: #6c757d;
                        font-size: 12px;
                    ">Test mode enabled - No real transactions will be processed</p>';
                    echo '</div>';
                    echo '</div>';
                }
                
                // Payment action button preview
                echo '<div class="payment-preview" style="
                    background: #f8f9fa;
                    border: 1px dashed #dee2e6;
                    border-radius: 8px;
                    padding: 16px;
                    text-align: center;
                    margin-bottom: 16px;
                ">';
                echo '<p style="
                    margin: 0 0 8px 0;
                    color: #6c757d;
                    font-size: 13px;
                    font-weight: 500;
                ">After clicking "Place Order", you will be redirected to:</p>';
                echo '<div style="
                    background: linear-gradient(135deg, #007bff, #0056b3);
                    color: white;
                    padding: 12px 24px;
                    border-radius: 25px;
                    display: inline-flex;
                    align-items: center;
                    gap: 8px;
                    font-weight: 600;
                    font-size: 14px;
                    box-shadow: 0 3px 10px rgba(0, 123, 255, 0.3);
                ">';
                echo '<span>🔐</span> ' . esc_html($site_info['gateway_name']) . ' Secure Checkout';
                echo '</div>';
                echo '</div>';
                
                // Trust indicators
                echo '<div class="trust-indicators" style="
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                    padding-top: 16px;
                    border-top: 1px solid #e9ecef;
                    font-size: 11px;
                    color: #6c757d;
                ">';
                echo '<div style="display: flex; align-items: center; gap: 4px;">
                    <span>🛡️</span> Bank-level encryption
                </div>';
                echo '<div style="display: flex; align-items: center; gap: 4px;">
                    <span>⚡</span> Instant processing
                </div>';
                echo '<div style="display: flex; align-items: center; gap: 4px;">
                    <span>🌍</span> Global acceptance
                </div>';
                echo '</div>';
                
                echo '</div>'; // End container
                
                // Add CSS animations
                echo '<style>
                    @keyframes pulse {
                        0% { opacity: 1; }
                        50% { opacity: 0.8; }
                        100% { opacity: 1; }
                    }
                    
                    .digikash-payment-container:hover {
                        transform: translateY(-2px);
                    }
                    
                    @media (max-width: 768px) {
                        .digikash-header {
                            flex-direction: column !important;
                            text-align: center !important;
                        }
                        
                        .feature-badges {
                            justify-content: center !important;
                        }
                        
                        .trust-indicators {
                            flex-direction: column !important;
                            gap: 8px !important;
                        }
                    }
                </style>';
            }
            
            /**
             * Get site info from DigiKash API for dynamic UI
             */
            public function get_site_info() {
                // Cache key for site info
                $cache_key = 'digikash_site_info_' . md5($this->api_base_url . $this->merchant_id);
                $cached_info = get_transient($cache_key);
                
                if ($cached_info !== false) {
                    return $cached_info;
                }
                
                // API call to get site info
                $environment = $this->testmode ? 'sandbox' : 'production';
                $args = array(
                    'method' => 'GET',
                    'timeout' => 10,
                    'headers' => array(
                        'Content-Type' => 'application/json',
                        'X-Environment' => $environment,
                        'X-Merchant-Key' => $this->merchant_id,
                        'X-API-Key' => $this->api_key
                    )
                );
                
                $response = wp_remote_get($this->api_base_url . '/api/v1/site-info', $args);
                
                if (is_wp_error($response)) {
                    error_log('DigiKash: Failed to fetch site info - ' . $response->get_error_message());
                    return $this->get_fallback_site_info();
                }
                
                $response_code = wp_remote_retrieve_response_code($response);
                if ($response_code !== 200) {
                    error_log('DigiKash: Site info API returned code: ' . $response_code);
                    return $this->get_fallback_site_info();
                }
                
                $body = wp_remote_retrieve_body($response);
                $data = json_decode($body, true);
                
                if (!$data || !isset($data['status']) || $data['status'] !== 'active') {
                    error_log('DigiKash: Invalid site info response');
                    return $this->get_fallback_site_info();
                }
                
                // Cache for 1 hour
                set_transient($cache_key, $data, HOUR_IN_SECONDS);
                
                return $data;
            }
            
            /**
             * Fallback site info when API fails
             */
            private function get_fallback_site_info() {
                return array(
                    'site_name' => 'DigiKash',
                    'site_logo' => 'https://digikash.coevs.com/images/2025-02-27_15-19-25_1365_6y61.png',
                    'gateway_name' => 'DigiKash Payment Gateway',
                    'gateway_description' => 'Secure payment powered by DigiKash',
                    'security_message' => 'Your payment information is processed securely. We do not store credit card details.',
                    'features' => array(
                        'ssl_secured' => true,
                        'instant_processing' => true,
                        'global_support' => true,
                        'mobile_ready' => true
                    ),
                    'branding' => array(
                        'powered_by_text' => 'Powered by DigiKash'
                    ),
                    'environments' => array(
                        'sandbox' => 'Test Mode - This is a test transaction. No real money will be charged'
                    )
                );
            }

            public function validate_fields() {
                error_log('DigiKash Gateway: validate_fields() called');
                return true;
            }

            public function process_payment($order_id) {
                error_log('DigiKash Gateway: process_payment() called for order ' . $order_id);
                
                $order = wc_get_order($order_id);
                if (!$order) {
                    wc_add_notice('Order not found.', 'error');
                    return array('result' => 'fail');
                }
                
                try {
                    // Prepare payment data for DigiKash API - exact format match
                    $payment_data = array(
                        'payment_amount' => (float) $order->get_total(),
                        'currency_code' => $order->get_currency(),
                        'ref_trx' => $order->get_order_number(),
                        'description' => 'Order #' . $order->get_order_number(),
                        'success_redirect' => $this->get_return_url($order),
                        'failure_url' => wc_get_checkout_url(),
                        'cancel_redirect' => wc_get_checkout_url(),
                        'ipn_url' => home_url('/wc-api/digikash_webhook')
                    );
                    
                    error_log('DigiKash Gateway: Initiating payment with data: ' . json_encode($payment_data));
                    
                    // Call DigiKash API to initiate payment
                    $response = wp_remote_post($this->api_base_url . '/api/v1/initiate-payment', array(
                        'method' => 'POST',
                        'timeout' => 30,
                        'headers' => array(
                            'Content-Type' => 'application/json',
                            'X-Environment' => $this->testmode ? 'sandbox' : 'production',
                            'X-Merchant-Key' => $this->merchant_id,
                            'X-API-Key' => $this->api_key
                        ),
                        'body' => json_encode($payment_data)
                    ));
                    
                    if (is_wp_error($response)) {
                        error_log('DigiKash Gateway: API request failed - ' . $response->get_error_message());
                        wc_add_notice('Payment processing failed. Please try again.', 'error');
                        return array('result' => 'fail');
                    }
                    
                    $response_body = wp_remote_retrieve_body($response);
                    $response_code = wp_remote_retrieve_response_code($response);
                    
                    error_log('DigiKash Gateway: API response code: ' . $response_code);
                    error_log('DigiKash Gateway: API response body: ' . $response_body);
                    
                    if ($response_code !== 200) {
                        wc_add_notice('Payment processing failed. Please try again.', 'error');
                        return array('result' => 'fail');
                    }
                    
                    $response_data = json_decode($response_body, true);
                    
                    if (!$response_data || !isset($response_data['payment_url'])) {
                        error_log('DigiKash Gateway: Invalid API response format');
                        wc_add_notice('Payment processing failed. Please try again.', 'error');
                        return array('result' => 'fail');
                    }
                    
                    $payment_url = $response_data['payment_url'];
                    
                    if (!filter_var($payment_url, FILTER_VALIDATE_URL)) {
                        error_log('DigiKash Gateway: Invalid payment URL received: ' . $payment_url);
                        wc_add_notice('Payment processing failed. Please try again.', 'error');
                        return array('result' => 'fail');
                    }
                    
                    // Update order status and save payment data
                    $order->update_status('pending', __('Awaiting DigiKash payment', 'woocommerce'));
                    $order->update_meta_data('_digikash_payment_data', $payment_data);
                    $order->update_meta_data('_digikash_payment_url', $payment_url);
                    $order->save();
                    
                    // Reduce stock levels
                    wc_reduce_stock_levels($order_id);
                    
                    // Empty cart
                    WC()->cart->empty_cart();
                    
                    error_log('DigiKash Gateway: Redirecting to payment URL: ' . $payment_url);
                    
                    // Return success with redirect to DigiKash payment URL
                    return array(
                        'result' => 'success',
                        'redirect' => $payment_url
                    );
                    
                } catch (Exception $e) {
                    error_log('DigiKash Gateway: Exception during payment processing: ' . $e->getMessage());
                    wc_add_notice('Payment processing error: ' . $e->getMessage(), 'error');
                    return array('result' => 'fail');
                }
            }

            public static function webhook_handler() {
                error_log('DigiKash Webhook: Handler called');
                error_log('DigiKash Webhook: Request method: ' . $_SERVER['REQUEST_METHOD']);
                error_log('DigiKash Webhook: Content type: ' . (isset($_SERVER['CONTENT_TYPE']) ? $_SERVER['CONTENT_TYPE'] : 'Not set'));
                
                // Get webhook headers
                $environment = isset($_SERVER['HTTP_X_ENVIRONMENT']) ? $_SERVER['HTTP_X_ENVIRONMENT'] : 'production';
                $signature = isset($_SERVER['HTTP_X_SIGNATURE']) ? $_SERVER['HTTP_X_SIGNATURE'] : '';
                $webhook_id = isset($_SERVER['HTTP_X_WEBHOOK_ID']) ? $_SERVER['HTTP_X_WEBHOOK_ID'] : '';
                
                error_log('DigiKash Webhook: Headers - Environment: ' . $environment . ', Signature: ' . ($signature ? 'present' : 'missing') . ', Webhook ID: ' . $webhook_id);
                
                // Get raw POST data
                $raw_post_data = file_get_contents('php://input');
                error_log('DigiKash Webhook: Raw POST data: ' . $raw_post_data);
                
                // Parse JSON payload
                $payload = json_decode($raw_post_data, true);
                if (!$payload) {
                    error_log('DigiKash Webhook: Failed to parse JSON payload');
                    status_header(400);
                    exit('Invalid JSON');
                }
                
                error_log('DigiKash Webhook: Parsed payload: ' . json_encode($payload));
                
                // Verify webhook signature for security
                if (!empty($this->webhook_secret)) {
                    $received_signature = isset($_SERVER['HTTP_X_SIGNATURE']) ? $_SERVER['HTTP_X_SIGNATURE'] : '';
                    if (empty($received_signature)) {
                        error_log('DigiKash Webhook: No signature provided but secret key is configured');
                        status_header(401);
                        exit('Signature required');
                    }
                    
                    // Calculate expected signature (HMAC-SHA256)
                    $expected_signature = 'sha256=' . hash_hmac('sha256', $raw_post_data, $this->webhook_secret);
                    
                    // Verify signature
                    if (!hash_equals($expected_signature, $received_signature)) {
                        error_log('DigiKash Webhook: Signature verification failed');
                        error_log('DigiKash Webhook: Expected: ' . $expected_signature);
                        error_log('DigiKash Webhook: Received: ' . $received_signature);
                        status_header(401);
                        exit('Invalid signature');
                    }
                    
                    error_log('DigiKash Webhook: Signature verified successfully');
                } else {
                    error_log('DigiKash Webhook: WARNING - No webhook secret configured, skipping signature verification');
                }
                
                // Extract webhook data according to DigiKash structure
                $webhook_status = isset($payload['status']) ? $payload['status'] : '';
                $webhook_message = isset($payload['message']) ? $payload['message'] : '';
                $webhook_timestamp = isset($payload['timestamp']) ? $payload['timestamp'] : time();
                $webhook_data = isset($payload['data']) ? $payload['data'] : [];
                
                // Extract transaction details from data object
                $ref_trx = isset($webhook_data['ref_trx']) ? $webhook_data['ref_trx'] : '';
                $amount = isset($webhook_data['amount']) ? $webhook_data['amount'] : 0;
                $currency_code = isset($webhook_data['currency_code']) ? $webhook_data['currency_code'] : '';
                $transaction_environment = isset($webhook_data['environment']) ? $webhook_data['environment'] : $environment;
                $is_sandbox = isset($webhook_data['is_sandbox']) ? $webhook_data['is_sandbox'] : ($environment === 'sandbox');
                
                error_log('DigiKash Webhook: Extracted data - ref_trx: ' . $ref_trx . ', status: ' . $webhook_status . ', amount: ' . $amount . ', environment: ' . $transaction_environment);
                
                // Validate required fields
                if (empty($ref_trx)) {
                    error_log('DigiKash Webhook: Missing ref_trx in webhook data');
                    status_header(400);
                    exit('Missing transaction reference');
                }
                
                if (empty($webhook_status)) {
                    error_log('DigiKash Webhook: Missing status in webhook data');
                    status_header(400);
                    exit('Missing payment status');
                }
                
                // Find the WooCommerce order
                $order_id = intval($ref_trx);
                $order = wc_get_order($order_id);
                
                if (!$order) {
                    error_log('DigiKash Webhook: Order not found for ref_trx: ' . $ref_trx);
                    status_header(404);
                    exit('Order not found');
                }
                
                error_log('DigiKash Webhook: Processing order #' . $order_id . ' with status: ' . $webhook_status);
                
                // Save webhook data to order meta for debugging
                $order->update_meta_data('_digikash_webhook_data', array(
                    'status' => $webhook_status,
                    'message' => $webhook_message,
                    'timestamp' => $webhook_timestamp,
                    'data' => $webhook_data,
                    'environment' => $transaction_environment,
                    'is_sandbox' => $is_sandbox,
                    'webhook_id' => $webhook_id,
                    'received_at' => current_time('mysql')
                ));
                
                // Add transaction reference if available
                if (!empty($webhook_data['transaction_id'])) {
                    $order->update_meta_data('_digikash_transaction_id', $webhook_data['transaction_id']);
                } elseif (!empty($ref_trx)) {
                    $order->update_meta_data('_digikash_transaction_id', $ref_trx);
                }
                
                // Handle different payment statuses from DigiKash
                switch (strtolower($webhook_status)) {
                    case 'completed':
                    case 'success':
                    case 'paid':
                        // Payment completed successfully
                        $order->payment_complete($ref_trx);
                        $order->add_order_note(__('DigiKash payment completed successfully via webhook. Transaction ID: ' . $ref_trx, 'woocommerce'));
                        error_log('DigiKash Webhook: Order #' . $order_id . ' marked as completed');
                        
                        // Add environment note for sandbox transactions
                        if ($is_sandbox) {
                            $order->add_order_note(__('⚠️ This was a sandbox/test transaction', 'woocommerce'));
                        }
                        break;
                        
                    case 'failed':
                    case 'error':
                    case 'declined':
                        // Payment failed
                        $order->update_status('failed', __('DigiKash payment failed: ' . $webhook_message, 'woocommerce'));
                        $order->add_order_note(__('DigiKash payment failed. Status: ' . $webhook_status . ', Message: ' . $webhook_message, 'woocommerce'));
                        error_log('DigiKash Webhook: Order #' . $order_id . ' marked as failed');
                        break;
                        
                    case 'cancelled':
                    case 'canceled':
                        // Payment cancelled by user
                        $order->update_status('cancelled', __('DigiKash payment cancelled by customer.', 'woocommerce'));
                        $order->add_order_note(__('DigiKash payment cancelled. Message: ' . $webhook_message, 'woocommerce'));
                        error_log('DigiKash Webhook: Order #' . $order_id . ' marked as cancelled');
                        break;
                        
                    case 'pending':
                    case 'processing':
                    case 'initiated':
                        // Payment is still processing
                        $order->update_status('on-hold', __('DigiKash payment is being processed: ' . $webhook_message, 'woocommerce'));
                        $order->add_order_note(__('DigiKash payment status update: ' . $webhook_status . ', Message: ' . $webhook_message, 'woocommerce'));
                        error_log('DigiKash Webhook: Order #' . $order_id . ' marked as on-hold (processing)');
                        break;
                        
                    case 'refunded':
                    case 'refund':
                        // Payment was refunded
                        $order->update_status('refunded', __('DigiKash payment refunded.', 'woocommerce'));
                        $order->add_order_note(__('DigiKash payment refunded. Message: ' . $webhook_message, 'woocommerce'));
                        error_log('DigiKash Webhook: Order #' . $order_id . ' marked as refunded');
                        break;
                        
                    default:
                        // Unknown status - log it but don't change order status
                        $order->add_order_note(__('DigiKash webhook received with unknown status: ' . $webhook_status . ', Message: ' . $webhook_message, 'woocommerce'));
                        error_log('DigiKash Webhook: Unknown status received for order #' . $order_id . ': ' . $webhook_status);
                        break;
                }
                
                // Save all changes to the order
                $order->save();
                
                error_log('DigiKash Webhook: Successfully processed webhook for order #' . $order_id);
                
                // Return success response
                status_header(200);
                exit('OK');
            }
        }
    } else {
        error_log('DigiKash Plugin: Class WC_Gateway_DigiKash already exists, skipping definition');
    }
    
    error_log('DigiKash Plugin: Adding gateway to WooCommerce');
    add_filter('woocommerce_payment_gateways', function($gateways) {
        error_log('DigiKash Plugin: woocommerce_payment_gateways filter called');
        $gateways[] = 'WC_Gateway_DigiKash';
        error_log('DigiKash Plugin: Added WC_Gateway_DigiKash to gateways array');
        return $gateways;
    });
}

// WooCommerce Blocks integration
add_action('woocommerce_blocks_loaded', function() {
    if (class_exists('Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType')) {
        require_once plugin_dir_path(__FILE__) . 'includes/class-wc-gateway-digikash-blocks.php';
        add_action('woocommerce_blocks_payment_method_type_registration', function($payment_method_registry) {
            $payment_method_registry->register(new WC_Gateway_DigiKash_Blocks());
        });
    }
});

// Force add to all available gateways for debugging
add_filter('woocommerce_available_payment_gateways', function($gateways) {
    error_log('DigiKash Plugin: woocommerce_available_payment_gateways filter called');
    if (!isset($gateways['digikash'])) {
        error_log('DigiKash Plugin: DigiKash not in available gateways, forcing add');
        if (class_exists('WC_Gateway_DigiKash')) {
            $gateways['digikash'] = new WC_Gateway_DigiKash();
            error_log('DigiKash Plugin: Forced DigiKash into available gateways');
        }
    } else {
        error_log('DigiKash Plugin: DigiKash already in available gateways');
    }
    return $gateways;
});

// Debug checkout context
add_action('woocommerce_checkout_init', function() {
    error_log('DigiKash Debug: woocommerce_checkout_init fired');
});

add_action('woocommerce_review_order_before_payment', function() {
    error_log('DigiKash Debug: woocommerce_review_order_before_payment fired');
});

// Debug: Check if WooCommerce is active
add_action('init', function() {
    if (class_exists('WooCommerce')) {
        error_log('DigiKash Plugin: WooCommerce is active');
        
        // Check if blocks checkout is being used
        if (function_exists('wc_get_page_id')) {
            $checkout_page_id = wc_get_page_id('checkout');
            $has_checkout_block = has_block('woocommerce/checkout', $checkout_page_id);
            error_log('DigiKash Plugin: Checkout page has blocks: ' . ($has_checkout_block ? 'YES' : 'NO'));
        }
    } else {
        error_log('DigiKash Plugin: WooCommerce is NOT active');
    }
});

// Plugin action links
add_filter('plugin_action_links_' . plugin_basename(__FILE__), function($links) {
    $settings_link = '<a href="admin.php?page=wc-settings&tab=checkout&section=digikash">Settings</a>';
    array_unshift($links, $settings_link);
    return $links;
});
